<?php

/*
 *  $Id: 7139361a6a95b8fee2720686c2526f06ca42a9c3 $
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the LGPL. For more information please see
 * <http://phing.info>.
 */

require_once 'phing/BuildFileTest.php';
require_once 'phing/tasks/system/PropertyTask.php';

/**
 * @author Hans Lellelid (Phing)
 * @author Conor MacNeill (Ant)
 * @package phing.tasks.system
 */
class PropertyTaskTest extends BuildFileTest
{

    public function setUp()
    {
        $this->configureProject(PHING_TEST_BASE . "/etc/tasks/property.xml");
    }

    public function test1()
    {
        // should get no output at all
        $this->expectOutputAndError("test1", "", "");
    }

    public function test2()
    {
        $this->expectLog("test2", "testprop1=aa, testprop3=xxyy, testprop4=aazz");
    }

    public function test4()
    {
        $this->expectLog("test4", "http.url is http://localhost:999");
    }

    public function testPrefixSuccess()
    {
        $this->executeTarget("prefix.success");
        $this->assertEquals("80", $this->project->getProperty("server1.http.port"));
    }

    public function testPrefixFailure()
    {
        try {
            $this->executeTarget("prefix.fail");
        } catch (BuildException $e) {
            $this->assertTrue(
                strpos($e->getMessage(), "Prefix is only valid") !== false,
                "Prefix allowed on non-resource/file load - "
            );

            return;
        }
        $this->fail("Did not throw exception on invalid use of prefix");
    }

    public function testFilterChain()
    {
        $this->executeTarget(__FUNCTION__);
        $this->assertEquals("World", $this->project->getProperty("filterchain.test"));
    }

    public function circularDefinitionTargets()
    {
        return array(
            array('test3'),
            array('testCircularDefinition1'),
            array('testCircularDefinition2'),
        );
    }

    /**
     * @dataProvider circularDefinitionTargets
     */
    public function testCircularDefinitionDetection($target)
    {
        try {
            $this->executeTarget($target);
        } catch (BuildException $e) {
            $this->assertTrue(
                strpos($e->getMessage(), "was circularly defined") !== false,
                "Circular definition not detected - "
            );

            return;
        }
        $this->fail("Did not throw exception on circular exception");
    }

    /**
     * Inspired by @link http://www.phing.info/trac/ticket/1118
     * This test should not throw exceptions
     */
    public function testUsingPropertyTwiceInPropertyValueShouldNotThrowException()
    {
        $this->executeTarget(__FUNCTION__);
    }
}

class HangDetectorPropertyTask extends PropertyTask
{

    protected function loadFile(PhingFile $file)
    {
        $props = new HangDetectorProperties();
        $props->load($file);
        $this->addProperties($props);
    }
}

class HangDetectorProperties extends Properties
{
    private $accesses = 0;

    public function getProperty($prop)
    {
        $this->accesses++;
        if ($this->accesses > 100) {
            throw new Exception('Cirular definition Hanged!');
        }

        return parent::getProperty($prop);
    }
}
