<?php

namespace App\Http\Controllers;

use App\Interfaces\UserRepositoryInterface;
use App\Validators\CreateUserValidator;
use App\Validators\EditUserValidator;
use Illuminate\Http\Request;
use Validator;
use Mail;
use Auth;

class UserController extends ApiController {

	/**
	 * @var UserRepositoryInterface
	 */
	private $user;

	/**
	 * @var CreateUserValidator
	 */
	private $createUserValidator;

	/**
	 * @var EditUserValidator
	 */
	private $editUserValidator;

	public function __construct(
		UserRepositoryInterface $user, 
		CreateUserValidator $createUserValidator,
		EditUserValidator $editUserValidator
	)
	{
		$this->user = $user;
		$this->createUserValidator = $createUserValidator;
		$this->editUserValidator = $editUserValidator;
	}

	/**
	 * Get All Users
	 * 
	 * @return mixed
	 */
	public function getAllUsers()
	{
		$users = $this->user->getAllUsers();
		return $users;
	}

	/**
	 * 
	 * Get single user
	 * 
	 * @param  int $id
	 * @return mixed
	 */
	public function getUser($id)
	{
		$user = $this->user->getUser($id);
		return $user;
	}

	/**
	 * Create new user
	 * 
	 * @return mixed
	 */
	public function createUser(Request $request)
	{

		if(! $this->createUserValidator->passes())
		{
			return $this->respondNotFound($this->createUserValidator->errors);
		}

		$data = [
			'first_name' => $request->input('first_name'),
			'last_name' => $request->input('last_name'),
			'username'  => $request->input('username'),
			'email' => $request->input('email'),
			'parent_email' => $request->input('parent_email'),
			'password' => bcrypt($request->input('password')),
			'grade_id' => $request->input('grade_id'),
			'school_id' => $request->input('school_id'),
			'course_id' => $request->input('course_id'),
			'address' => $request->input('address'),
			'city' => $request->input('city'),
			'state' => $request->input('state'),
			'country' => $request->input('country'),
			'home_phone' => $request->input('home_phone'),
			'parent_phone' => $request->input('parent_phone'),
			'zip' => $request->input('zip'),
		];

		$user = $this->user->createUser($data);

		if($user)
		{
			Mail::send('emails.confirmation', ['user' => $user], function($m) use ($user)
			{
				$m->to($user->email, $user->first_name)
				  ->subject('The Ivy Key - Email Confirmation');
			});

			return view('thankyou');
		}
	}

	/**
	 * Edit user
	 * 
	 * @param  int $id
	 * @return mixed
	 */
	public function editUser(Request $request)
	{
		$user = Auth::user();

		$data = [
			'first_name' => $request->input('first_name'),
			'last_name' => $request->input('last_name'),
			'username'  => $request->input('username'),
			'email'     => $request->input('email'),
			'parent_email' => $request->input('parent_email'),
			'password' => bcrypt($request->input('password')),
			'grade_id' => $request->input('grade_id'),
			'school_id' => $request->input('school_id'),
			'course_id' => $request->input('course_id'),
			'address' => $request->input('address'),
			'city' => $request->input('city'),
			'state' => $request->input('state'),
			'country' => $request->input('country'),
			'home_phone' => $request->input('home_phone'),
			'parent_phone' => $request->input('parent_phone'),
			'zip' => $request->input('zip'),
		];
		
		if(! $this->editUserValidator->passes())
		{
			return $this->respondNotFound($this->editUserValidator->errors);
		}

		$validator = Validator::make($request->all(), [
			'email' => 'required|email|unique:users,email,'.$user->id,
		]);

		if(!$validator->passes())
		{
			return $validator->errors();
		}

		$user = $this->user->editUser($user->id, $data);

		if(!$user)
		{
			return $this->respondNotFound('User not found.');
		}

		return back()->with('message', 'Profile updated.');

	}

	/**
	 * Delete user
	 * 
	 * @param  int $id
	 * @return mixed
	 */
	public function deleteUser($id)
	{
		//check if id is numeric
		if(! is_numeric($id))
		{
			return $this->respondWrongArgs();
		}

		$user = $this->user->deleteUser($id);

		if(! $user)
		{
			return $this->respondNotFound('User not found.');
		}

		return $this->respondSuccess('User deleted.');
	}

}
