<?php

namespace App\Http\Controllers;

use App\Program;
use App\User;
use Validator;
use Auth;
use Mail;
use AuthorizeNetAIM;
use Illuminate\Http\Request;
use App\Interfaces\PaymentRepositoryInterface;

class PaymentController extends ApiController {
    
    /**
     * @var PaymentRepositoryInterface
     */
    private $payment;

    public function __construct(PaymentRepositoryInterface $payment)
    {
        $this->payment = $payment;
    }

    /**
     * Index page
     * 
     * @return view
     */
    public function index()
    {
        
        return view('payment');
    }

    /**
     * Post payment
     * 
     * @return mixed
     */
    public function postPayment(Request $request)
    {
        $progId = $request->input('program');
        $program = Program::whereId($progId)->first();
        $groupId = $program->groupId;

        //check if user is logged in
        if(Auth::check())
        {

            $userId = Auth::user()->id;

            $auth = new AuthorizeNetAIM(\Config::get('authorize.apiLogin'), \Config::get('authorize.transId'));
            $auth->setSandbox(true);
            $auth->amount = $program->price;
            $auth->email = Auth::user()->email;
            $auth->card_num = $request->input('cc-number');
            $auth->exp_date = $request->input('exp-date');

            $response = $auth->authorizeAndCapture();

            if($response->approved)
            {
                // set the program_id for the user
                $user = User::whereId($userId)->first();
                $user->programs()->attach($progId);

                return redirect('/ivykey/dashboard/classroom')->with('message', 'Great! New class added.');
            }
            else
            {
                // redirect user back with error message
                return back()->with('message', $response->error_message)->withInput();
            }
        }
        else
        {   
            $data = [
                'first_name'  => $request->input('first_name'),
                'last_name'   => $request->input('last_name'),
                'username'    => $request->input('username'),
                'parent_email'     => $request->input('parent_email'),
                'parent_phone'    => $request->input('parent_phone'),
                'password'    => $request->input('password'),
                'password_confirmation' => $request->input('password_confirmation'),
                'program_id'  => $request->input('program'),
                'parent_email'     => $request->input('parent_email')
            ];
            
            $validator = Validator::make($data, [
                'first_name' => 'sometimes|required|max:30',
                'last_name'  => 'sometimes|required',
                'email'      => 'sometimes|required|email',
                'username'   => 'sometimes|required|alpha_num|max:255',
                'parent_name'     => 'sometimes|required|email',
                'parent_phone'     => 'sometimes|required',
                'program_id' => 'required|numeric',
                'password'   => 'sometimes|required|min:6|confirmed',
                'password_confirmation'  => 'sometimes|required|min:6',
            ]);

            if($validator->fails())
            {
                return redirect()->back()->withInput()->with('message', 'Please, check all fields again.');
            }
            else
            {
                $auth = new AuthorizeNetAIM(\Config::get('authorize.apiLogin'), \Config::get('authorize.transId'));
                $auth->setSandbox(true);
                $auth->amount = $program->price;
                $auth->email = $request->input('email');
                $auth->card_num = $request->input('cc-number');
                $auth->exp_date = $request->input('exp-date');

                $response = $auth->authorizeAndCapture();

                if($response->approved)
                {
                    // create new user
                    $user = new User();
                    $user->first_name = $request->input('first_name');
                    $user->last_name = $request->input('last_name');
                    $user->email = $request->input('email');
                    $user->username = $request->input('username');
                    $user->parent_email = $request->input('parent_email');
                    $user->parent_phone = $request->input('parent_phone');
                    $user->password = bcrypt($request->input('password'));
                    $user->code = str_random();

                    if($user->save())
                    {
                        // send to pivot
                        $user->programs()->attach($progId);
                        $user->groups()->attach($groupId);
                        // send account confirmation email
                        Mail::send('emails.confirmation', ['user' => $user], function($m) use ($user)
                        {
                            $m->to($user->email, $user->first_name)
                              ->subject('The Ivy Key - Email Confirmation');
                        });

                        // send payment invoice to user's email
                    }

                    // redirect user back to the 'thank you for your payment' page
                    return view('/thankyounote');
                }
                else 
                {
                    // redirect user back with error message
                    return back()->with('message', $response->error_message)->withInput();
                }
            }

        }
    

    }

    /**
     * Account activation
     * 
     * @param  string $code
     * @return view
     */
    public function activateUser($code)
    {
        $user = $this->payment->activateUser($code);

        if(!$user)
        {
            return back()->with('message', $this->respondNotFound('User email already confirmed!'));
        }

        return view('activatedAccount');
    }

    /**
     * Enroll
     * 
     * @return mixed
     */
    public function enroll(Request $request)
    {
        $progId = $request->input('program');
        // check if user is logged in
        if(Auth::check())
        {
            $userId = Auth::user()->id;

            $user = User::whereId($userId)->first();
            // send to pivot
            $user->programs()->attach($progId);

            return redirect('/ivykey/dashboard/classroom')->with('message', 'Great! New class added');
        }
        else
        {   
            $url = \URL::to("/enroll?");
            // register user than redirect back to desired program
            return redirect("/register?return=$url" . "program=" .$progId);
        }
    }

}